function [LU_endogenous, LU_exogenous, years, ForestAge_endogenous, ScrubAge_endogenous ] = LUAM2_spatial_land_allocation(c, LandUseChange, Co2price, LURNZDATA, forestHR, harvest_age )
%
% Land Use Allocation Module - function 2 (LUAM2)
%
% This file spatially allocates changes in land use.
%
% If dairy increases:
    % the most suitable sheep, forestry and scrub land changes to dairy
    % limited such that this may not cause forest land to decrease further
    % than specified by LandUseChange
% If dairy decreases:
    % the least suitable dairy land changes to sheep
    % and may change land use again due to changes in  sheep land
% If sheep increases:
    % the most suitable forest and scrub land changes to sheep
    % limited such that this may not cause forest land to decrease further
    % than specified by LandUseChange
% If sheep decreases:
    % the least suitable sheep land changes to scrub
    % and may change land use again due to changes in scrub land
% If forest increases:
    % the most suitable scrub land changes to forest land
% If forest decreases:
    % the least suitable forest land changes to scrub land
%

% Code by Simon Anastasiadis : 2011-11-03

%% Parameters


%% Exogenous Land Use
% Identify and separate out

% identify existing exogenous land
% Horticulture land           =  5
% Non-productive land         =  6
% Urban land                  =  7
% Other animals and lifestyle =  8
% Indigenous forest           =  9
% Pasture on public land      = 10
% DoC and public land         = 11
iExogenous = LURNZDATA(:,c.LUType) >= 5;
% exogenous land remains unchanged each iteration so exclude
LU_exogenous = LURNZDATA(iExogenous,[c.sorter, c.LUType]);

%% Endogenous Land Use
% Identify and separate out
% identify existing endogenous land
% dairy farm land         = 1
% sheep & beef farm land  = 2
% plantaion forestry      = 3
% scrub land              = 4
iEndogenous = LURNZDATA(:,c.LUType) <= 4;
% separate endogenous land for analysis
LURNZendogenous = LURNZDATA(iEndogenous,:);

%% Setup storage for results

% Storage for endogenous land uses for each year
LU_endogenous = zeros(size(LURNZendogenous,1),size(LandUseChange,1)+2);
% fill with sorters
LU_endogenous(:,1) = LURNZendogenous(:,c.sorter);
% fill with initial land use
LU_endogenous(:,2) = LURNZendogenous(:,c.LUType);

% Storage for Forestry age
ForestAge_endogenous = zeros(size(LURNZendogenous,1),size(LandUseChange,1)+2);
% fill with sorters
ForestAge_endogenous(:,1) = LURNZendogenous(:,c.sorter);
% fill with initial ages
ForestAge_endogenous(:,2) = LURNZendogenous(:,c.forestAge);

% Storage for Scrub Age
ScrubAge_endogenous = zeros(size(LURNZendogenous,1),size(LandUseChange,1)+2);
% fill with sorters
ScrubAge_endogenous(:,1) = LURNZendogenous(:,c.sorter);
% fill with initial ages
ScrubAge_endogenous(:,2) = LURNZendogenous(:,c.newScrub);


%% Add New columns for allocation routine

% new simulated land use
c.NewLU    = size(LURNZendogenous,2)+1;
% sorter for pixels that could change land use
c.TempSort = size(LURNZendogenous,2)+2;

LURNZendogenous(:,[c.NewLU,c.TempSort]) = 0;

%% Iterate over the simulation years

fprintf(' - - - Simulating: Year, Pixel changes for  Dairy,  Sheep, Forest,  Scrub\n')

for ii = 1:size(LandUseChange,1)
    % simulation year
    year = LandUseChange(ii,1);
    
    %% Prevent Pre-1990 LUCAS forest being converted
    %
    % We do not want Pre-1990 forest to change land use if there is a
    % positive carbon price.
    % Avoid this by temporarially setting pre1990 forest to LandUse -3 so
    % it will not change land use in the allocation algorithm.
    %
    % 2012-03-26
    % At Levi's request: we do not want Pre-1990 forest to ever change land
    % use (for comparability with other scenarios).
    
%     if Co2price ~= 0
        % identify pre90 forest
%         iPre90 = LURNZendogenous(:,c.LUType) == 3 & LURNZendogenous(:,c.LUCAS) == 72;
        % change pre90 forest land use value to -3
%         LURNZendogenous(iPre90,c.LUType) = -3;
%     end
    
    %% Prevent deforestation of young forests
    %
    % Only forestry aged 26 - 32 years is considered harvestable. Hence we
    % prevent young forest from being converted to other land uses
    
    % identify young forest
    iYoung = LURNZendogenous(:,c.LUType) == 3 & LURNZendogenous(:,c.forestAge) < 26;
    % allow harvested areas that are awaiting replacing to be converted
    iYoung = iYoung & LURNZendogenous(:,c.forestAge) ~= 0;
    % change young forest land use value to -3
    LURNZendogenous(iYoung,c.LUType) = -3;
    
    %% Changes in land use
    
    % change in land use
    dDairy  = LandUseChange(ii, 2);
    dSheep  = LandUseChange(ii, 3);
    dForest = LandUseChange(ii, 4);
    dScrub  = LandUseChange(ii, 5);
    
    % assert total change in Land Use = 0
    dLandUse = dDairy + dSheep + dForest + dScrub;
    msg = sprintf(['Error in LUAM2_spatial_allocation\n'...
                   'Non zero total change in land use\n'...
                   '%12.2f change instead'],dLandUse);
    assert(dLandUse == 0, msg);
    
    %% Display Changes in land use

    % this string is formatted in this way to line up / be consistent with
    % the heading given in line 82.
    fprintf('                   %4d                    %6d  %6d  %6d  %6d\n', year, dDairy, dSheep, dForest, dScrub);
    
    %% 1. Dairy Allocation
    
    % existing dairy land remains dairy land unless changed below
    iLand = LURNZendogenous(:,c.LUType)==1 & LURNZendogenous(:,c.NewLU)==0;
    LURNZendogenous(iLand,c.NewLU) = 1;
    
    if dDairy > 0
        % handle the increase in dairy land
        [LURNZendogenous, dSheep, dForest, dScrub] =...
            increase_dairy_land(LURNZendogenous, c, dDairy, dSheep, dForest, dScrub);
    elseif dDairy < 0
        % handle the decrease in dairy land
        [LURNZendogenous, dSheep, dForest, dScrub] =...
            decrease_dairy_land(LURNZendogenous, c, dDairy, dSheep, dForest, dScrub);
    end
     
    %% 2. Sheep & Beef Allocation
    
    % existing sheep land remains sheep land unless changed above or below
    iLand = LURNZendogenous(:,c.LUType)==2 & LURNZendogenous(:,c.NewLU)==0;
    LURNZendogenous(iLand,c.NewLU) = 2;
    
    if dSheep > 0
        % handle the increase in sheep & beef land
        [LURNZendogenous, dForest, dScrub] =...
            increase_sheep_land(LURNZendogenous, c, dSheep, dForest, dScrub);
    elseif dSheep < 0
        % handle the decrase in sheep & beef land
        [LURNZendogenous, dForest, dScrub] =...
            decrease_sheep_land(LURNZendogenous, c, dSheep, dForest, dScrub);
    end
        
    %% 3. Plantation Forest Allocation
    
    % existing forest land remains forest unless changed above or below
    iLand = LURNZendogenous(:,c.LUType)==3 & LURNZendogenous(:,c.NewLU)==0;
    LURNZendogenous(iLand,c.NewLU) = 3;
    
    if dForest > 0
        % handle the increase in forest land
        [LURNZendogenous, dScrub] =...
            increase_forest_land(LURNZendogenous, c, dForest, dScrub);
    elseif dForest < 0
        % handle the decrase in forest land
        [LURNZendogenous, dScrub] =...
            decrease_forest_land(LURNZendogenous, c, dForest, dScrub);
    end
    
    %% 4. Scrub Land Allocation
    
    % existing scrub land remains scrub unless changed above
    iLand = LURNZendogenous(:,c.LUType)==4 & LURNZendogenous(:,c.NewLU)==0;
    LURNZendogenous(iLand,c.NewLU) = 4;
    
    % dScrub should now equal zero
    msg = sprintf(['Error in LUAM2_spatial_allocation\n',...
                'Dairy, sheep and forest lands changed but scrub fails']);
    assert(dScrub==0,msg)
    
    % alternatively, in previous versions of LURNZ there might be 1 or 2
    % pixels of scrub land that needed to change at this point. In this
    % case we allowed for changes in land use between scrub and "Other
    % animals and lifestyle land" (c.LUType = 8).
    %
    % This should no longer be necessary.
    
    %% Return LURNZ data to original order
    LURNZendogenous = sortrows(LURNZendogenous,c.sorter);
    
    %% Resolve Pre-1990 LUCAS forest
    %
    % We make the LandUse code for Pre1990 forest -3 temporarially so that
    % it will not change land use in the above code.
    % Here we restore it to its original value.
    
%     if Co2price ~= 0
        % change pre90 forest land use value to 3
%         LURNZendogenous(iPre90,c.LUType) = 3;
        % and NewLU as well
%         LURNZendogenous(iPre90,c.NewLU) = 3;
%     end
    
    %% Resolve Young forests
    %
    % We make the LandUse code for young forest -3 temporarially so that it
    % will not change land use in the above code.
    % Here we restore it to its original value.
    
    % change young forest LandUse value to 3
    LURNZendogenous(iYoung,c.LUType) = 3;
    % and NewLU as well
    LURNZendogenous(iYoung,c.NewLU) = 3;
    
    %% Handle Changes in Forestry Ages
    
    % we need to update forestry age
    % and store this information somewhere
    
    % Update Forestry Ages
    [F_newage, F_newage2] = UpdateForestryAge(LURNZendogenous, c, forestHR, harvest_age);
    % Insert into LURNZendogenous
    LURNZendogenous(:,c.forestAge) = F_newage;
    
    %% Handle Changes in Scrub Ages
    
    % we need to update scrub age
    % and store this information somewhere
    
    % Update Scrub Ages
    S_newage = UpdateScrubAge(LURNZendogenous, c);
    % Insert into LURNZendogenous
    LURNZendogenous(:,c.newScrub) = S_newage;
    
    %% Save results
    
    % save new Land Use to LU_endogenous
    LU_endogenous(:,ii+2) = LURNZendogenous(:,c.NewLU);
    
    % save forestry age to ForestAge_endogenous
    ForestAge_endogenous(:,ii+2) = F_newage2;
    
    % save scrub age to ScrubAge_endogenous
    ScrubAge_endogenous(:,ii+2) = S_newage;
    
    %% Prepare for next iteration
    
    LURNZendogenous(:,c.LUType) = LURNZendogenous(:,c.NewLU);
    LURNZendogenous(:,c.NewLU) = 0;
    
end

%% Construct a vector of simulation years

% construct empty matrix
years = zeros(size(LandUseChange,1)+1,1);
% fill matrix
years(1,1)     = LandUseChange(1,1)-1;
years(2:end,1) = LandUseChange(:,1);

%% Inform User projecting land use change is complete

msg = sprintf(' - - LUAM2 : spatial allocation of changes complete');
disp(msg)

end

%% Subfunction : Handle INcreases in dairy land

function [LURNZendogenous, dSheep, dForest, dScrub] =...
            increase_dairy_land(LURNZendogenous, c, dDairy, dSheep, dForest, dScrub)

% select all sheep, forest and scrub land without a new land use
iLand = (LURNZendogenous(:,c.LUType)==2 |...
         LURNZendogenous(:,c.LUType)==3 |...
         LURNZendogenous(:,c.LUType)==4 )...
       & LURNZendogenous(:,c.NewLU)==0;
% set a temporary sorter
iProb = LURNZendogenous(:,c.uDairy) ~= 0;
LURNZendogenous(iLand & iProb,c.TempSort) = 1;

% check there is enough available land to convert to dairy
if sum(iLand & iProb) < dDairy
    shortfall = dDairy - sum(iLand & iProb);
    warning('LUAM2TA:negativeLandUse',...
           ['Warning from LUAM2_spatial_allocation\n'...
            'Increase in dairy land > current available land\n'...
            '%d shortfall discarded'],shortfall);
    % Adjust for shortfall in land
    dDairy = dDairy - shortfall;
    dSheep = dSheep + shortfall;
end

% sort rows by temp sorter and pixel probabilities
LURNZendogenous = sortrows(LURNZendogenous,[-c.TempSort,-c.uDairy]);

% Require that the decrease in forest due to increases in dairy land does
% not exceed the total decrease in forest land (and if forest land
% increases then no forest land may be converted to dairy land)

% identify forestry pixels currently going to change land use
iForest = LURNZendogenous(1:dDairy,c.LUType)==3;
% if too much forestry land will be conveted
if sum(iForest) > -dForest
    % mark forestry land that we want to keep
    iForest = LURNZendogenous(:,c.LUType)==3;
    % record order of forestry pixels
    iForestOrder = cumsum(iForest);
    % change temp sorter to exclude forest pixels that should not change
    LURNZendogenous(iForest & iForestOrder > -dForest,c.TempSort) = 0;
    
    % re-check there is enough available land to convert to sheep & beef
    if sum(LURNZendogenous(:,c.TempSort)) < dDairy
        shortfall = dDairy - sum(LURNZendogenous(:,c.TempSort));
        warning('LUAM2TA:negativeLandUse',...
               ['Warning from LUAM2_spatial_allocation\n'...
                'Increase in dairy land > current available land\n'...
                '%d shortfall discarded'],shortfall);
        % Adjust for shortfall in land
        dDairy = dDairy - shortfall;
        dSheep = dSheep + shortfall;
    end
    
    % resort
    LURNZendogenous = sortrows(LURNZendogenous,[-c.TempSort,-c.uDairy]);
end

% top most dDairy pixels will change land use
LURNZendogenous(1:dDairy,c.NewLU) = 1;

% count change in pixels & update
i = LURNZendogenous(1:dDairy,c.LUType)==2; dSheep  = dSheep  + sum(i);
i = LURNZendogenous(1:dDairy,c.LUType)==3; dForest = dForest + sum(i);
i = LURNZendogenous(1:dDairy,c.LUType)==4; dScrub  = dScrub  + sum(i);

% Reset temp sorter
LURNZendogenous(:,c.TempSort) = 0;

end


%% Subfunction : Handle DEcreases in dairy land

function [LURNZendogenous, dSheep, dForest, dScrub] =...
            decrease_dairy_land(LURNZendogenous, c, dDairy, dSheep, dForest, dScrub)

% select all dairy land
iLand = LURNZendogenous(:,c.LUType)==1 | LURNZendogenous(:,c.NewLU)==1;
% set a temporary sorter
iProb = LURNZendogenous(:,c.uDairy) ~= 0;
LURNZendogenous(iLand & iProb,c.TempSort) = 1;

% check there is enough dairy land to give away
if sum(iLand & iProb) < -dDairy
    shortfall = -dDairy - sum(iLand & iProb);
    warning('LUAM2TA:negativeLandUse',...
           ['Warning from LUAM2_spatial_allocation\n'...
            'Decrease in dairy land > current dairy land\n'...
            '%d shortfall discarded'],shortfall);
    % Adjust for shortfall in land
    dDairy = dDairy + shortfall;
    dSheep = dSheep - shortfall;
end
    
% sort rows by temp sorter and pixel probabilities
LURNZendogenous = sortrows(LURNZendogenous,[-c.TempSort, c.uDairy]);
% top most dDairy pixels will change land use
LURNZendogenous(1:(-dDairy),c.NewLU) = 2;

% count change in pixels & update
dSheep = dSheep + dDairy;

% Reset temp sorter
LURNZendogenous(:,c.TempSort) = 0;

end

%% Subfunction : Handle INcreases in sheep & beef land

function [LURNZendogenous, dForest, dScrub] =...
            increase_sheep_land(LURNZendogenous, c, dSheep, dForest, dScrub)

% select all forest and scrub land without a new land use
iLand = (LURNZendogenous(:,c.LUType)==3 |...
         LURNZendogenous(:,c.LUType)==4 )...
       & LURNZendogenous(:,c.NewLU)==0;
%set a temporarary sorter
iProb = LURNZendogenous(:,c.uSheep) ~= 0;
LURNZendogenous(iLand & iProb,c.TempSort) = 1;

% check there is enough available land to convert to sheep & beef
if sum(iLand & iProb) < dSheep
    shortfall = dSheep - sum(iLand & iProb);
    warning('LUAM2TA:negativeLandUse',...
           ['Warning from LUAM2_spatial_allocation\n'...
            'Increase in sheep land > current available land\n'...
            '%d shortfall discarded'],shortfall);
    % Adjust for shortfall in land
    dSheep = dSheep - shortfall;
    dScrub = dScrub + shortfall;
end

% sort rows by temp sorter and pixel probabilities
LURNZendogenous = sortrows(LURNZendogenous,[-c.TempSort,-c.uSheep]);

% Require that the decrease in forest due to increases in sheep land does
% not exceed the total decrease in forest land (and if forest land
% increases then no forest land may be converted to sheep & beef land)

% identify forestry pixels currently going to change land use
iForest = LURNZendogenous(1:dSheep,c.LUType)==3;
% if too much forestry land will be conveted
if sum(iForest) > -dForest
    % mark forestry land that we want to keep
    iForest = LURNZendogenous(:,c.LUType)==3;
    % record order of forestry pixels
    iForestOrder = cumsum(iForest);
    % change temp sorter to exclude forest pixels that should not change
    LURNZendogenous(iForest & iForestOrder > -dForest,c.TempSort) = 0;
    
    % re-check there is enough available land to convert to sheep & beef
    if sum(LURNZendogenous(:,c.TempSort)) < dSheep
        shortfall = dSheep - sum(LURNZendogenous(:,c.TempSort));
        warning('LUAM2TA:negativeLandUse',...
               ['Warning from LUAM2_spatial_allocation\n'...
                'Increase in sheep land > current available land\n'...
                '%d shortfall discarded'],shortfall);
        % Adjust for shortfall in land
        dSheep = dSheep - shortfall;
        dScrub = dScrub + shortfall;
    end
    
    % resort
    LURNZendogenous = sortrows(LURNZendogenous,[-c.TempSort,-c.uSheep]);
end

% top most dSheep pixels will change land use
LURNZendogenous(1:dSheep,c.NewLU) = 2;

% count change in pixels & update
i = LURNZendogenous(1:dSheep,c.LUType)==3; dForest = dForest + sum(i);
i = LURNZendogenous(1:dSheep,c.LUType)==4; dScrub  = dScrub  + sum(i);

% Reset temp sorter
LURNZendogenous(:,c.TempSort) = 0;

end

%% Subfunction : Handle DEcreases in sheep & beef land

function [LURNZendogenous, dForest, dScrub] =...
            decrease_sheep_land(LURNZendogenous, c, dSheep, dForest, dScrub)

% select all sheep & beef land (exclude land that was dairy)
iLand = LURNZendogenous(:,c.NewLU)==2 & LURNZendogenous(:,c.LUType)~=1;
% set a temporary sorter
iProb = LURNZendogenous(:,c.uSheep) ~= 0;
LURNZendogenous(iLand & iProb,c.TempSort) = 1;

% check there is enough sheep & beef land to give away
if sum(iLand & iProb) < -dSheep
    shortfall = -dSheep - sum(iLand & iProb);
    warning('LUAM2TA:negativeLandUse',...
           ['Warning from LUAM2_spatial_allocation\n'...
            'Decrease in sheep land > current sheep land\n'...
            '%d shortfall discarded'],shortfall);
    % Adjust for shortfall in land
    dSheep = dSheep + shortfall;
    dScrub = dScrub - shortfall;
end

% sort rows by temp sorter and pixel probabilities
LURNZendogenous = sortrows(LURNZendogenous,[-c.TempSort, c.uSheep]);
% top most dSheep pixels will change land use
LURNZendogenous(1:(-dSheep),c.NewLU) = 4;

% count change in pixels & update
dScrub = dScrub + dSheep;

% Reset temp sorter
LURNZendogenous(:,c.TempSort) = 0;

end

%% Subfunction : Handle INcreases in plantation forestry land

function [LURNZendogenous, dScrub] =...
            increase_forest_land(LURNZendogenous, c, dForest, dScrub)

% select all scrub land without a new land use
iLand = (LURNZendogenous(:,c.LUType)==4 & LURNZendogenous(:,c.NewLU)==0) |...
         LURNZendogenous(:,c.NewLU)==4;
%set a temporarary sorter
iProb = LURNZendogenous(:,c.uForest) ~= 0;
LURNZendogenous(iLand & iProb,c.TempSort) = 1;

% check there is enough available land to convert to forest
if sum(iLand & iProb) < dForest
    shortfall = dForest - sum(iLand & iProb);
    warning('LUAM2TA:negativeLandUse',...
           ['Warning from LUAM2_spatial_allocation\n'...
            'Increase in forest land > current available land\n'...
            '%d shortfall discarded'],shortfall);
    % Adjust for shortfall in land
    dForest = dForest - shortfall;
    dScrub  = dScrub  + shortfall;
end

% sort rows by temp sorter and pixel probabilities
LURNZendogenous = sortrows(LURNZendogenous,[-c.TempSort,-c.uForest]);
% top most dSheep pixels will change land use
LURNZendogenous(1:dForest,c.NewLU) = 3;

% count change in pixels & update
i = LURNZendogenous(1:dForest,c.LUType)==4; dScrub = dScrub  + sum(i);
i = LURNZendogenous(1:dForest,c.LUType)==2; dScrub = dScrub  + sum(i);

% Reset temp sorter
LURNZendogenous(:,c.TempSort) = 0;

end

%% Subfunction : Handle DEcreases in plantation forestry land

function [LURNZendogenous, dScrub] =...
            decrease_forest_land(LURNZendogenous, c, dForest, dScrub)

% select all forest land
iLand = LURNZendogenous(:,c.NewLU)==3;
% set a temporary sorter
iProb = LURNZendogenous(:,c.uForest) ~= 0;
LURNZendogenous(iLand & iProb,c.TempSort) = 1;

% check there is enough forest land to give away
if sum(iLand & iProb) < -dForest
    shortfall = -dForest - sum(iLand & iProb);
    warning('LUAM2TA:negativeLandUse',...
           ['Warning from LUAM2_spatial_allocation\n'...
            'Decrease in forest land > current forest land\n'...
            '%d shortfall discarded'],shortfall);
    % Adjust for shortfall in land
    dForest = dForest + shortfall;
    dScrub  = dScrub  - shortfall;
end

% sort rows by temp sorter and pixel probabilities
LURNZendogenous = sortrows(LURNZendogenous,[-c.TempSort, c.uForest]);
% top most dForest pixels will change land use
LURNZendogenous(1:(-dForest),c.NewLU) = 4;

% count change in pixels & update
dScrub = dScrub + dForest;

% Reset temp sorter
LURNZendogenous(:,c.TempSort) = 0;
    
end

%% Subfunction : Update Forestry Ages

function [newage, newage2] = UpdateForestryAge(LURNZendogenous, c, forestHR, harvest_age)
%
% Subfunction UpdateForestryAge
%
% Function takes the forestry age data and steps it forward a single year.
% This includes land use conversions, harvesting forestry, and maturing
% forestry.
%
% Two vectors of age are given as output:
% 'newage' gives the age of all forestry pixels (without negative numbers)
% 'newage2' gives the age of forestry pixels and uses negative numbers to
% indicate harvest and deforestation (for use in the GHG module).
%
% We received LURNZendogenous already sorted according to sortrows

% check LURNZendogenous is sorted
msg = sprintf('Error in LUAM2:\nLURNZendogenous must be sorted before updating forest ages');
assert(all(LURNZendogenous(1:end-1,c.sorter) < LURNZendogenous(2:end,c.sorter)),msg);

% for ease produce a matrix with col [1, 2, 3] = [sorter, old LU, new LU]
LU_old_new = LURNZendogenous(:,[c.sorter, c.LUType, c.NewLU]);

% For each year
% Calculate new ages:
%   new forestry = age 1
%   harvest and deforestation using indicators
%   remaining forest ages 1 year
%   remaining forest may be subject to harvest
%   age zero forest may be planted = age 1
% Convert ages indicating harvest to zero
% Convert ages indicating deforestation to -9999 (the null value)

% Identify new forestry
iNF = LU_old_new(:,2) ~= 3 & LU_old_new(:,3) == 3;
% new forestry has age 1
LURNZendogenous(iNF,c.forestAge) = 1;

% Identify deforested land (moving out of forestry)
iFN = LU_old_new(:,2) == 3 & LU_old_new(:,3) ~= 3;
% allow deforested land to mature by 1 year
LURNZendogenous(iFN,c.forestAge) = LURNZendogenous(iFN,c.forestAge) + 1;
% use negative of age to indicate deforestation
LURNZendogenous(iFN,c.forestAge) = -LURNZendogenous(iFN,c.forestAge);

% Identify forestry land that is maturing
% forestry older than 80 stays classified as age 80
iFF = LU_old_new(:,2) == 3 & LU_old_new(:,3) == 3 & LURNZendogenous(:,c.forestAge) ~= 0 & LURNZendogenous(:,c.forestAge) ~= 80;
% existing forest ages by 1 year
LURNZendogenous(iFF,c.forestAge) = LURNZendogenous(iFF,c.forestAge) + 1;

% Harvest
% Total available area for harvest
iHarvestable = (26 + harvest_age) <= LURNZendogenous(:,c.forestAge) & LURNZendogenous(:,c.forestAge) <= (32 + harvest_age) & LU_old_new(:,3) == 3;
areaHarvestable = sum(iHarvestable) - sum(iFN);
% Total area harvested
areaHarvested = areaHarvestable * forestHR.harvest_ed_able_ratio;
% Area harvested by forestry age
areaByAge = [forestHR.harvest_by_age(:,1) , areaHarvested * forestHR.harvest_by_age(:,2)];
% Allocate harvest spatially
LURNZendogenous(iHarvestable,c.forestAge) = Spatial_Harvest(LURNZendogenous(iHarvestable,:), areaByAge, c);
% note we only pass the harvestable pixels to the subfunction

% Replant
% Total available area for replant
iReplantable = LURNZendogenous(:,c.forestAge) == 0 & LU_old_new(:,3) == 3;
areaReplantable = sum(iReplantable);
% Total area replanted
areaReplanted = round(areaReplantable * forestHR.percent_replace);
% Allocate replanting spatially
LURNZendogenous(iReplantable,c.forestAge) = Spatial_Replant(LURNZendogenous(iReplantable,:), areaReplanted, c);
% note we only pass the replantable pixels to the subfunction

% Extract newage2
newage2 = LURNZendogenous(:,c.forestAge);

% Convert negative ages used to indicate harvest to zero
iZF = LURNZendogenous(:,c.forestAge) < 0 & LU_old_new(:,3) == 3;
% Land is awaiting harvesting
LURNZendogenous(iZF,c.forestAge) = 0;

% Convert negative ages used to indicate deforestation to null
iZN = LURNZendogenous(:,c.forestAge) < 0 & LU_old_new(:,3) ~= 3 & LURNZendogenous(:,c.forestAge) ~= -9999;
% Land is not forestry and as no age
LURNZendogenous(iZN,c.forestAge) = -9999;

% Extract newage
newage = LURNZendogenous(:,c.forestAge);

end

%% Subfunction : Allocate Harvest Spatially

function [agesPostHarvest ] = Spatial_Harvest(LURNZendogenous, areaByAge, c)
% Given the number of pixels harvested we allocate these pixels spatially
% Wei Zhang's algorithm used forestry profitablility and prioritised the
% most profitable pixels. We replace forest profitability with pixel
% probabilities.

% Assemble matrix for calculation
% sorter age LUType uForest temp
harvest_grid = [LURNZendogenous(:,c.sorter), LURNZendogenous(:,c.forestAge), LURNZendogenous(:,c.LUType), LURNZendogenous(:,c.uForest)];
harvest_grid(:,end+1) = 0;

% loop through harvest ages
for ii = 1:size(areaByAge,1)
    % set indicator for pixels of the right age
    harvest_grid(:,5) = harvest_grid(:,2) == areaByAge(ii,1) & harvest_grid(:,3) == 3;
    % area harvested = min(harvest target, available pixels)
    dHarvest = min(sum(harvest_grid(:,5)) , round(areaByAge(ii,2)));
    
    % sort pixels
    harvest_grid = sortrows(harvest_grid,[-5, -4]);
    % top most 'dHarvest' pixels are harvested
    harvest_grid(1:dHarvest,2) = -harvest_grid(1:dHarvest,2);
    
end

% sort rows
harvest_grid = sortrows(harvest_grid,1);

% age
agesPostHarvest = harvest_grid(:,2);

end

%% Subfunction : Allocate Replanting Spatially

function [agesPostReplant ] = Spatial_Replant(LURNZendogenous, areaReplanted, c)
% Given the number of pixels replanted we allocate these pixels spatially
% Wei Zhang's algorithm used forestry profitablility and prioritised the
% most profitable pixels. We replace forest profitability with pixel
% probabilities.

% Assemble matrix for calculation
% sorter age uForest temp
replant_grid = [LURNZendogenous(:,c.sorter), LURNZendogenous(:,c.forestAge), LURNZendogenous(:,c.uForest)];
replant_grid(:,end+1) = 0;

% set indicator for pixels of the right age
replant_grid(:,4) = replant_grid(:,2) == 0;
% sort pixels
replant_grid = sortrows(replant_grid,[-4, -3]);
% top most 'areaReplanted' pixels are harvested
replant_grid(1:areaReplanted,2) = 1;

% sort rows
replant_grid = sortrows(replant_grid,1);

% age
agesPostReplant = replant_grid(:,2);

end

%% Subfunction : Update Scrub Ages

function [newage] = UpdateScrubAge(LURNZendogenous, c)
%
% Subfunction UpdateScrubAge
%
% Function takes the scrub age data and steps it forward a single year.
% This includes land use conversions, and maturing forestry.
%
% Two vectors of age are given as output:
% 'newage' gives the age of all scrub pixels (without negative numbers)
% We do not require negative numbers for scrub ages to indicate scrub
% emissions.
%
% We received LURNZendogenous already sorted according to sortrows

% check LURNZendogenous is sorted
msg = sprintf('Error in LUAM2:\nLURNZendogenous must be sorted before updating forest ages');
assert(all(LURNZendogenous(1:end-1,c.sorter) < LURNZendogenous(2:end,c.sorter)),msg);

% for ease produce a matrix with col [1, 2, 3] = [sorter, old LU, new LU]
LU_old_new = LURNZendogenous(:,[c.sorter, c.LUType, c.NewLU]);

% For each year
% Calculate new ages:
%   new scrub = age 1
%   existing scrub ages 1 year
%   cleared scrub = -9999 (the null value)

% new scrub planting
iNS = LU_old_new(:,3)==4 & LU_old_new(:,2)~=4;
LURNZendogenous(iNS,c.newScrub) = 1;

% scrub aging
iSS = LU_old_new(:,3)==4 & LU_old_new(:,2)==4;
% pixel must already be at least age 1
% otherwise, old scrub can also age, which could cause problems
iAge = LURNZendogenous(:,c.newScrub) > 0;
LURNZendogenous(iSS & iAge,c.newScrub) = LURNZendogenous(iSS & iAge,c.newScrub) + 1;

% scrub clearing
iSN = LU_old_new(:,3)~=4 & LU_old_new(:,2)==4;
LURNZendogenous(iSN,c.newScrub) = -9999;

% Extract newage
newage = LURNZendogenous(:,c.newScrub);

end

